import numpy as np
from numpy.random import default_rng
import matplotlib.pyplot as plt
import math
import itertools

# Initialize the random seed.

rng = default_rng()

# randInstance:       Generates a single-item instance with a random vector x.
#
# Returns:
#   The vector x of realization probabilities.
#
# Parameters:
#   n:                Number of elements.

def randInstance(n):
    nums = []
    for i in range(n):
        nums.append(rng.random())
    k = sum(nums)
    for i in range(n):
        nums[i] = nums[i] / k
    return nums

# uniInstance:       Generates a uniform single-item instance.
#
# Returns:
#   The vector x of realization probabilities.
#
# Parameters:
#   n:                Number of elements.
#   k:		      Number equal to the sum of all x_i's. For the single-item setting, this is 1.

def uniInstance(n, k):
    return [k / n] * n

# flip:		      Flips a coin with probability p.
#
# Returns:
#   1 if the coin toss was successful, 0 otherwise.
#
# Parameters:
#   p:                Probability of coin toss being 1.

def flip(p):
    return 1 if rng.random() < p else 0

# singleOCRS:	      Runs both our greedy OCRS and the FSZ greedy OCRS for a given single-item instance.
#
# Returns:
#   Minimum and Average selectability of our algorithm for the given instance.
#   Minimum and Average selectability of the FSZ algorithm for the given instance.
#
# Parameters:
#   instance:         Vector x of realization probabilities for each element.
#   iterations:       Number of iterations to run each instance for to estimate selectability.

def singleOCRS(instance, iterations):
    n = len(instance)
    selectability = [0] * n
    selectabilityFSZ = [0] * n
    active_iters = [0] * n

    for j in range(iterations):
        cap = 0
        capFSZ = 0
        feasible = [flip((1 - math.exp(-instance[i])) / instance[i]) for i in range(n)]
        feasibleFSZ = [flip(0.5) for i in range(n)]
        active = [flip(instance[i]) for i in range(n)]
        for i in range(n):
            if(active[i] == 1):
                active_iters[i] += 1
                if(feasible[i] == 1 and cap == 0):
                    selectability[i] += 1
                    cap = 1
                if(feasibleFSZ[i] == 1 and capFSZ == 0):
                    selectabilityFSZ[i] += 1
                    capFSZ = 1

    results = [0 for i in range(n)]
    resultsFSZ = [0 for i in range(n)]
    minim = 1
    minimFSZ = 1
    for i in range(n):
        if(active_iters[i] != 0):
            results[i] = selectability[i] / active_iters[i]
            resultsFSZ[i] = selectabilityFSZ[i] / active_iters[i]
            if(results[i] < minim):
                minim = results[i]
            if(resultsFSZ[i] < minimFSZ):
                minimFSZ = resultsFSZ[i]
    avg = sum(results) / n
    avgFSZ = sum(resultsFSZ) / n
    return (minim, avg, minimFSZ, avgFSZ)

# feasibleTrans:      Determines if a given set of elements is feasible in a given transversal matroid.
#
# Returns:
#   True if ''augmentedSel'' is feasible in the transversal matroid ''feasible'', False otherwise.
#
# Parameters:
#   augmentedSel:     Set of elements, given as a list.
#   v:		      Number of right-side vertices in the bipartite graph of the given transversal matroid.
#   feasible:	      Transversal matroid, given as a list of lists of size v x n where feasible[k][i] == 1,
#		         if the edge {i,k} exists in the transversal matroid, and feasible[k][i] == 0 otherwise.

def feasibleTrans(augmentedSel, v, feasible):
    for perm in itertools.permutations(list(range(v)), len(augmentedSel)):
        pairs = list(zip(list(perm), augmentedSel))
        s = 0
        for pair in pairs:
            if(feasible[pair[0]][pair[1]]):
                s += 1
        if(s == len(augmentedSel)):
            return True
    return False

# transOCRS:	      Runs both our greedy OCRS and the FSZ greedy OCRS for a given transversal matroid instance.
#
# Returns:
#   Minimum and Average selectability of our algorithm for the given instance.
#   Minimum and Average selectability of the FSZ algorithm for the given instance.
#
# Parameters:
#   instance:         Vector x of realization probabilities for each element.
#   v:		      Number of right-side vertices in the bipartite graph of the given transversal matroid.
#   iterations:       Number of iterations to run each instance for to estimate selectability.

def transOCRS(instance, v, iterations):
    n = len(instance)
    selectability = [0] * n
    selectabilityFSZ = [0] * n
    active_iters = [0] * n

    for j in range(iterations):
        sel = []
        capFSZ = 0
        feasible = [[flip( 1 - (pow(1 - (1 - math.exp(-instance[i])) / instance[i], 1.0 / v)) ) for i in range(n)] for k in range(v)]
        feasibleFSZ = [flip(0.5) for i in range(n)]
        active = [flip(instance[i]) for i in range(n)]
        for i in range(n):
            if(active[i] == 1):
                active_iters[i] += 1
                if(feasibleFSZ[i] == 1 and capFSZ < v):
                    selectabilityFSZ[i] += 1
                    capFSZ += 1
                if(len(sel) < v and feasibleTrans(sel+[i], v, feasible)):
                    selectability[i] += 1
                    sel += [i]

    results = [0 for i in range(n)]
    resultsFSZ = [0 for i in range(n)]
    minim = 1
    minimFSZ = 1
    for i in range(n):
        if(active_iters[i] != 0):
            results[i] = selectability[i] / active_iters[i]
            resultsFSZ[i] = selectabilityFSZ[i] / active_iters[i]
            if(results[i] < minim):
                minim = results[i]
            if(resultsFSZ[i] < minimFSZ):
                minimFSZ = resultsFSZ[i]
    avg = sum(results) / n
    avgFSZ = sum(resultsFSZ) / n
    return (minim, avg, minimFSZ, avgFSZ)


# runsSingle:       Executes experiments for the single-item setting.
#
# Returns:
#   Minimum and Average selectability of our algorithm for the uniform instance.
#   Minimum and Average selectability of our algorithm for a randomly-generated instance.
#   Minimum and Average selectability of the FSZ algorithm for the uniform instance.
#   Minimum and Average selectability of the FSZ algorithm for a randomly-generated instance.
#
# Parameters:
#   n:                Number of elements.
#   iterations:       Number of iterations to run each instance for to estimate selectability.

def runsSingle(n, iterations):

    # Single-item setting, x: uniform instance, y: random instance
    x = uniInstance(n, 1.0)
    y = randInstance(n)

    (minX, avgX, minXFSZ, avgXFSZ) = singleOCRS(x, iterations)
    (minY, avgY, minYFSZ, avgYFSZ) = singleOCRS(y, iterations)

    return [minX, avgX, minY, avgY, minXFSZ, avgXFSZ, minYFSZ, avgYFSZ]
 
# runsTrans:       Executes experiments for the transversal matroid setting.
#
# Returns:
#   Minimum and Average selectability of our algorithm for the uniform transversal matroid instance.
#   Minimum and Average selectability of the FSZ algorithm for the Uniform instance.
#   Minimum and Average selectability of the FSZ algorithm for the uniform transversal matroid instance.
#
# Parameters:
#   n:                Number of elements.
#   v:		      Number of right-side vertices in the bipartite graph of the transversal matroid.
#   iterations:       Number of iterations to run each instance for to estimate selectability.

def runsTrans(n, v, iterations):

    # Transversal matroid setting, x: uniform instance
    x = uniInstance(n, v)

    (minX, avgX, minXFSZ, avgXFSZ) = transOCRS(x, v, iterations)

    return [minX, avgX, minXFSZ, avgXFSZ]

# plotSingle:	      Generates the plots for the single-item instance.
#
# Returns:
#   Nothing, the plots are saved as files.
#
# Parameters:
#   ns:               List of number of elements to run the experiments for.
#   iterations:       Number of iterations to run each instance for to estimate selectability.
#   repetitions:      Number of times the entire experiment is ran from scratch to account for errors.

def plotSingle(ns, iterations, repetitions):

    minUni = []
    avgUni = []
    minRand = []
    avgRand = []
    errLowMinUni = []
    errLowAvgUni = []
    errLowMinRand = []
    errLowAvgRand = []
    errUppMinUni = []
    errUppAvgUni = []
    errUppMinRand = []
    errUppAvgRand = []

    minUniFSZ = []
    avgUniFSZ = []
    minRandFSZ = []
    avgRandFSZ = []
    errLowMinUniFSZ = []
    errLowAvgUniFSZ = []
    errLowMinRandFSZ = []
    errLowAvgRandFSZ = []
    errUppMinUniFSZ = []
    errUppAvgUniFSZ = []
    errUppMinRandFSZ = []
    errUppAvgRandFSZ = []

    for n in ns:
        minUniPerRepetition = []
        avgUniPerRepetition = []
        minRandPerRepetition = []
        avgRandPerRepetition = []
        minUniPerRepetitionFSZ = []
        avgUniPerRepetitionFSZ = []
        minRandPerRepetitionFSZ = []
        avgRandPerRepetitionFSZ = []

        for i in range(repetitions):
            currentN = runsSingle(n, iterations)
            minUniPerRepetition.append(currentN[0])
            avgUniPerRepetition.append(currentN[1])
            minRandPerRepetition.append(currentN[2])
            avgRandPerRepetition.append(currentN[3])
            minUniPerRepetitionFSZ.append(currentN[4])
            avgUniPerRepetitionFSZ.append(currentN[5])
            minRandPerRepetitionFSZ.append(currentN[6])
            avgRandPerRepetitionFSZ.append(currentN[7])
        
        minUni.append(sum(minUniPerRepetition) / repetitions)
        avgUni.append(sum(avgUniPerRepetition) / repetitions)
        minRand.append(sum(minRandPerRepetition) / repetitions)
        avgRand.append(sum(avgRandPerRepetition) / repetitions)
        errLowMinUni.append(sum(minUniPerRepetition) / repetitions - min(minUniPerRepetition))
        errLowAvgUni.append(sum(avgUniPerRepetition) / repetitions - min(avgUniPerRepetition))
        errLowMinRand.append(sum(minRandPerRepetition) / repetitions - min(minRandPerRepetition))
        errLowAvgRand.append(sum(avgRandPerRepetition) / repetitions - min(avgRandPerRepetition))
        errUppMinUni.append(max(minUniPerRepetition) - sum(minUniPerRepetition) / repetitions)
        errUppAvgUni.append(max(avgUniPerRepetition) - sum(avgUniPerRepetition) / repetitions)
        errUppMinRand.append(max(minRandPerRepetition) - sum(minRandPerRepetition) / repetitions)
        errUppAvgRand.append(max(avgRandPerRepetition) - sum(avgRandPerRepetition) / repetitions)

        minUniFSZ.append(sum(minUniPerRepetitionFSZ) / repetitions)
        avgUniFSZ.append(sum(avgUniPerRepetitionFSZ) / repetitions)
        minRandFSZ.append(sum(minRandPerRepetitionFSZ) / repetitions)
        avgRandFSZ.append(sum(avgRandPerRepetitionFSZ) / repetitions)
        errLowMinUniFSZ.append(sum(minUniPerRepetitionFSZ) / repetitions - min(minUniPerRepetitionFSZ))
        errLowAvgUniFSZ.append(sum(avgUniPerRepetitionFSZ) / repetitions - min(avgUniPerRepetitionFSZ))
        errLowMinRandFSZ.append(sum(minRandPerRepetitionFSZ) / repetitions - min(minRandPerRepetitionFSZ))
        errLowAvgRandFSZ.append(sum(avgRandPerRepetitionFSZ) / repetitions - min(avgRandPerRepetitionFSZ))
        errUppMinUniFSZ.append(max(minUniPerRepetitionFSZ) - sum(minUniPerRepetitionFSZ) / repetitions)
        errUppAvgUniFSZ.append(max(avgUniPerRepetitionFSZ) - sum(avgUniPerRepetitionFSZ) / repetitions)
        errUppMinRandFSZ.append(max(minRandPerRepetitionFSZ) - sum(minRandPerRepetitionFSZ) / repetitions)
        errUppAvgRandFSZ.append(max(avgRandPerRepetitionFSZ) - sum(avgRandPerRepetitionFSZ) / repetitions)

    fig1, (ax0, ax1) = plt.subplots(nrows=2, sharex=True)
    fig2, (ax2, ax3) = plt.subplots(nrows=2, sharex=True)
    ax0.errorbar(np.array(ns), np.array(minUni), yerr=[np.array(errLowMinUni), np.array(errUppMinUni)], fmt='o-r', capsize=2.0, capthick = 4.0, label='Our Algorithm')
    ax0.errorbar(np.array(ns), np.array(minUniFSZ), yerr=[np.array(errLowMinUniFSZ), np.array(errUppMinUniFSZ)], fmt='o-b', capsize=2.0, capthick = 4.0, label='[FSZ16]')
    ax0.set(ylim=(0.0, 0.5))
    ax0.legend(loc='best')
    ax0.set_title('Minimum Selectability - Uniform Instance')
    ax1.errorbar(np.array(ns), np.array(avgUni), yerr=[np.array(errLowAvgUni), np.array(errUppAvgUni)], fmt='s--r', capsize=2.0, capthick = 4.0, label='Our Algorithm')
    ax1.errorbar(np.array(ns), np.array(avgUniFSZ), yerr=[np.array(errLowAvgUniFSZ), np.array(errUppAvgUniFSZ)], fmt='s--b', capsize=2.0, capthick = 4.0, label='[FSZ16]')
    ax1.set(ylim=(0.3, 0.75))
    ax1.legend(loc='best')
    ax1.set_title('Average Selectability - Uniform Instance')
    s1 = "single-item-uni.png"
    fig1.savefig(s1)

    ax2.errorbar(np.array(ns), np.array(minRand), yerr=[np.array(errLowMinRand), np.array(errUppMinRand)], fmt='o-r', capsize=2.0, capthick = 4.0, label='Our Algorithm')
    ax2.errorbar(np.array(ns), np.array(minRandFSZ), yerr=[np.array(errLowMinRandFSZ), np.array(errUppMinRandFSZ)], fmt='o-b', capsize=2.0, capthick = 4.0, label='[FSZ16]')
    ax2.set(ylim=(0.0, 0.5))
    ax2.legend(loc='best')
    ax2.set_title('Minimum Selectability - Random Instance')
    ax3.errorbar(np.array(ns), np.array(avgRand), yerr=[np.array(errLowAvgRand), np.array(errUppAvgRand)], fmt='s--r', capsize=2.0, capthick = 4.0, label='Our Algorithm')
    ax3.errorbar(np.array(ns), np.array(avgRandFSZ), yerr=[np.array(errLowAvgRandFSZ), np.array(errUppAvgRandFSZ)], fmt='s--b', capsize=2.0, capthick = 4.0, label='[FSZ16]')
    ax3.set(ylim=(0.3, 0.75))
    ax3.legend(loc='best')
    ax3.set_title('Average Selectability - Random Instance')
    s2 = "single-item-rand.png"
    fig2.savefig(s2)

# plotTrans:	      Generates the plots for the transversal matroid instance.
#
# Returns:
#   Nothing, the plots are saved as files.
#
# Parameters:
#   n:                Number of elements.
#   neighbors:	      List of number of right-side vertices in the bipartite graph of the transversal
#			matroid to run the experiment for.
#   iterations:       Number of iterations to run each instance for to estimate selectability.
#   repetitions:      Number of times the entire experiment is ran from scratch to account for errors.

def plotTrans(n, neighbors, iterations, repetitions):

    minUni = []
    avgUni = []
    errLowMinUni = []
    errLowAvgUni = []
    errUppMinUni = []
    errUppAvgUni = []

    minUniFSZ = []
    avgUniFSZ = []
    errLowMinUniFSZ = []
    errLowAvgUniFSZ = []
    errUppMinUniFSZ = []
    errUppAvgUniFSZ = []

    for v in neighbors:
        minUniPerRepetition = []
        avgUniPerRepetition = []
        minUniPerRepetitionFSZ = []
        avgUniPerRepetitionFSZ = []

        for i in range(repetitions):
            currentRep = runsTrans(n, v, iterations)
            minUniPerRepetition.append(currentRep[0])
            avgUniPerRepetition.append(currentRep[1])
            minUniPerRepetitionFSZ.append(currentRep[2])
            avgUniPerRepetitionFSZ.append(currentRep[3])
        
        minUni.append(sum(minUniPerRepetition) / repetitions)
        avgUni.append(sum(avgUniPerRepetition) / repetitions)
        errLowMinUni.append(sum(minUniPerRepetition) / repetitions - min(minUniPerRepetition))
        errLowAvgUni.append(sum(avgUniPerRepetition) / repetitions - min(avgUniPerRepetition))
        errUppMinUni.append(max(minUniPerRepetition) - sum(minUniPerRepetition) / repetitions)
        errUppAvgUni.append(max(avgUniPerRepetition) - sum(avgUniPerRepetition) / repetitions)

        minUniFSZ.append(sum(minUniPerRepetitionFSZ) / repetitions)
        avgUniFSZ.append(sum(avgUniPerRepetitionFSZ) / repetitions)
        errLowMinUniFSZ.append(sum(minUniPerRepetitionFSZ) / repetitions - min(minUniPerRepetitionFSZ))
        errLowAvgUniFSZ.append(sum(avgUniPerRepetitionFSZ) / repetitions - min(avgUniPerRepetitionFSZ))
        errUppMinUniFSZ.append(max(minUniPerRepetitionFSZ) - sum(minUniPerRepetitionFSZ) / repetitions)
        errUppAvgUniFSZ.append(max(avgUniPerRepetitionFSZ) - sum(avgUniPerRepetitionFSZ) / repetitions)

    fig1, (ax0, ax1) = plt.subplots(nrows=2, sharex=True)
    ax0.errorbar(np.array(neighbors), np.array(minUni), yerr=[np.array(errLowMinUni), np.array(errUppMinUni)], fmt='o-r', capsize=2.0, capthick = 4.0, label='Our Algorithm')
    ax0.errorbar(np.array(neighbors), np.array(minUniFSZ), yerr=[np.array(errLowMinUniFSZ), np.array(errUppMinUniFSZ)], fmt='o-b', capsize=2.0, capthick = 4.0, label='[FSZ16]')
    ax0.set(ylim=(0.0, 0.7), xticks=neighbors)
    ax0.legend(loc='best')
    ax0.set_title('Minimum Selectability - Transversal Matroid')
    ax1.errorbar(np.array(neighbors), np.array(avgUni), yerr=[np.array(errLowAvgUni), np.array(errUppAvgUni)], fmt='s--r', capsize=2.0, capthick = 4.0, label='Our Algorithm')
    ax1.errorbar(np.array(neighbors), np.array(avgUniFSZ), yerr=[np.array(errLowAvgUniFSZ), np.array(errUppAvgUniFSZ)], fmt='s--b', capsize=2.0, capthick = 4.0, label='[FSZ16]')
    ax1.set(ylim=(0.0, 1.0))
    ax1.legend(loc='best')
    ax1.set_title('Average Selectability - Transversal Matroid')
    s = "trans.png"
    fig1.savefig(s)
